﻿using System.Collections.Specialized;
using System.Net;

public static class WebRequestor
{
    static WebRequestor()
    {
        ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
    }

    public static string Get(string requestUrl, string operation, string accept = WebConstants.JsonMimeType)
    {
        var request = WebRequestorCore.CreateRequest(requestUrl);

        request.KeepAlive = true;
        request.Method = "GET";
        request.Accept = accept;

        return WebRequestorCore.GetResponse(request, operation);
    }

    public static string Post(string requestUrl, byte[] data, string operation, NameValueCollection headersCollection = null)
    {
        return WriteRequest(requestUrl, "POST", data, WebConstants.JsonMimeType, operation, headersCollection);
    }

    public static string Post(string requestUrl, byte[] data, string contentType, string operation, NameValueCollection headersCollection = null)
    {
        return WriteRequest(requestUrl, "POST", data, contentType, operation, headersCollection);
    }

    public static string Put(string requestUrl, byte[] data, string operation, NameValueCollection headersCollection = null)
    {
        return WriteRequest(requestUrl, "PUT", data, WebConstants.JsonMimeType, operation, headersCollection);
    }

    public static string Put(string requestUrl, byte[] data, string contentType, string operation, NameValueCollection headersCollection = null)
    {
        return WriteRequest(requestUrl, "PUT", data, contentType, operation, headersCollection);
    }

    public static string Delete(string requestUrl, string operation, NameValueCollection headersCollection = null)
    {
        return WriteRequest(requestUrl, "DELETE", null, null, operation, headersCollection);
    }

    private static string WriteRequest(string requestUrl, string method, byte[] data, string contentType, string operation, NameValueCollection headersCollection = null)
    {
        var request = WebRequestorCore.CreateRequest(requestUrl);

        request.KeepAlive = true;
        request.Method = method;

        if (contentType != null)
            request.ContentType = contentType;

        if (headersCollection != null && headersCollection.HasKeys())
            request.Headers.Add(headersCollection);

        if (data != null)
        {
            request.ContentLength = data.Length;

            using (var dataStream = request.GetRequestStream())
            {
                dataStream.Write(data, 0, data.Length);
            }
        }

        return WebRequestorCore.GetResponse(request, operation);
    }
}
